<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    xmlns:w="http://schemas.openxmlformats.org/wordprocessingml/2006/main"
    xmlns:pic="http://schemas.openxmlformats.org/drawingml/2006/picture"
    xmlns:a="http://schemas.openxmlformats.org/drawingml/2006/main"
    xmlns:r="http://schemas.openxmlformats.org/officeDocument/2006/relationships"
    exclude-result-prefixes="xs pic r a"
    version="1.0">
    <xsl:output method="text"/>
    
    <!-- Parameter to control whether to expand images into markdown image references -->
    <xsl:param name="expandImages" select="'true'"/>
    
    <!-- Variables to track list state -->
    <xsl:variable name="list-indent" select="'  '"/>
    
    <!-- Template to get heading level from paragraph style -->
    <xsl:template name="get-heading-level">
        <xsl:param name="style-name"/>
        <xsl:choose>
            <xsl:when test="contains($style-name, 'Heading1') or contains($style-name, 'heading1') or contains($style-name, 'Title')">1</xsl:when>
            <xsl:when test="contains($style-name, 'Heading2') or contains($style-name, 'heading2')">2</xsl:when>
            <xsl:when test="contains($style-name, 'Heading3') or contains($style-name, 'heading3')">3</xsl:when>
            <xsl:when test="contains($style-name, 'Heading4') or contains($style-name, 'heading4')">4</xsl:when>
            <xsl:when test="contains($style-name, 'Heading5') or contains($style-name, 'heading5')">5</xsl:when>
            <xsl:when test="contains($style-name, 'Heading6') or contains($style-name, 'heading6')">6</xsl:when>
            <xsl:otherwise>0</xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Template to generate markdown heading markers -->
    <xsl:template name="heading-markers">
        <xsl:param name="level"/>
        <xsl:if test="$level > 0">
            <xsl:text>#</xsl:text>
            <xsl:if test="$level > 1">
                <xsl:call-template name="heading-markers">
                    <xsl:with-param name="level" select="$level - 1"/>
                </xsl:call-template>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle paragraphs in table cells without styles - no extra newlines -->
    <xsl:template match="w:tc//w:p[not(w:pPr/w:pStyle)]">
        <xsl:apply-templates select="node()"/>
    </xsl:template>
    
    <!-- Handle ListNumber paragraphs in table cells as numbered lists -->
    <xsl:template match="w:tc//w:p[w:pPr/w:pStyle[@w:val='ListNumber']]">
        <xsl:variable name="ilvl" select="number(w:pPr/w:numPr/w:ilvl/@w:val)"/>
        <!-- Add indentation based on list level (2 spaces per level) -->
        <xsl:if test="$ilvl > 0">
            <xsl:call-template name="repeat-string">
                <xsl:with-param name="string" select="$list-indent"/>
                <xsl:with-param name="count" select="$ilvl"/>
            </xsl:call-template>
        </xsl:if>
        <!-- Use numbered list format (1. ) - markdown will auto-number sequential items -->
        <xsl:text>1. </xsl:text>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
    </xsl:template>
    
    <!-- Handle paragraphs with styles in table cells (excluding ListNumber which is handled separately) -->
    <xsl:template match="w:tc//w:p[w:pPr/w:pStyle and not(w:pPr/w:pStyle[@w:val='ListNumber'])]">
        <xsl:variable name="style-name" select="w:pPr/w:pStyle/@w:val"/>
        <xsl:variable name="heading-level">
            <xsl:call-template name="get-heading-level">
                <xsl:with-param name="style-name" select="$style-name"/>
            </xsl:call-template>
        </xsl:variable>
        <xsl:if test="$heading-level > 0">
            <xsl:call-template name="heading-markers">
                <xsl:with-param name="level" select="$heading-level"/>
            </xsl:call-template>
            <xsl:text> </xsl:text>
        </xsl:if>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
    </xsl:template>
    
    <!-- Handle paragraphs with heading styles (takes priority over lists) - but not TOC, Caption, ListNumber, ListBullet, HTMLPreformatted, or in table cells -->
    <xsl:template match="w:p[w:pPr/w:pStyle 
        and not(w:pPr/w:pStyle[starts-with(@w:val, 'TOC')]) 
        and not(w:pPr/w:pStyle[@w:val='Caption']) 
        and not(w:pPr/w:pStyle[@w:val='ListNumber']) 
        and not(w:pPr/w:pStyle[@w:val='ListBullet']) 
        and not(w:pPr/w:pStyle[@w:val='HTMLPreformatted']) 
        and not(w:pPr/w:pStyle[contains(@w:val, 'List') or contains(@w:val, 'Bullet')]) 
        and not(ancestor::w:tc)]">  
        <xsl:variable name="style-name" select="w:pPr/w:pStyle/@w:val"/>
        <xsl:variable name="heading-level">
            <xsl:call-template name="get-heading-level">
                <xsl:with-param name="style-name" select="$style-name"/>
            </xsl:call-template>
        </xsl:variable>
        <xsl:text>

</xsl:text>
        <xsl:if test="$heading-level > 0">
            <xsl:call-template name="heading-markers">
                <xsl:with-param name="level" select="$heading-level"/>
            </xsl:call-template>
            <xsl:text> </xsl:text>
        </xsl:if>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
        <xsl:if test="$heading-level > 0">
            <xsl:text>

</xsl:text>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle paragraphs with numbering (lists) - but not if they're headings -->
    <xsl:template match="w:p[w:pPr/w:numPr and not(w:pPr/w:pStyle)]">
        <xsl:variable name="ilvl" select="number(w:pPr/w:numPr/w:ilvl/@w:val)"/>
        <xsl:text>

</xsl:text>
        <!-- Add indentation based on list level (2 spaces per level) -->
        <xsl:if test="$ilvl > 0">
            <xsl:call-template name="repeat-string">
                <xsl:with-param name="string" select="$list-indent"/>
                <xsl:with-param name="count" select="$ilvl"/>
            </xsl:call-template>
        </xsl:if>
        <!-- Use numbered list format (1. ) - in real Word docs, we'd need to check numbering format -->
        <xsl:text>1. </xsl:text>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
    </xsl:template>
    
    <!-- Handle paragraphs with ListNumber style as numbered lists - but not if they're headings -->
    <xsl:template match="w:p[w:pPr/w:pStyle[@w:val='ListNumber'] and not(ancestor::w:tc)]">
        <xsl:variable name="ilvl" select="number(w:pPr/w:numPr/w:ilvl/@w:val)"/>
        <xsl:text>
</xsl:text>
        <!-- Add indentation based on list level (2 spaces per level) -->
        <xsl:if test="$ilvl > 0">
            <xsl:call-template name="repeat-string">
                <xsl:with-param name="string" select="$list-indent"/>
                <xsl:with-param name="count" select="$ilvl"/>
            </xsl:call-template>
        </xsl:if>
        <!-- Use numbered list format (1. ) - markdown will auto-number sequential items -->
        <xsl:text>1. </xsl:text>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
    </xsl:template>
    
    <!-- Handle paragraphs with ListBullet style as unordered lists - but not if they're headings or in table cells -->
    <xsl:template match="w:p[w:pPr/w:pStyle[@w:val='ListBullet'] and not(ancestor::w:tc)]">
        <xsl:variable name="ilvl" select="number(w:pPr/w:numPr/w:ilvl/@w:val)"/>
        <xsl:text>
</xsl:text>
        <!-- Add indentation based on list level (2 spaces per level) -->
        <xsl:if test="$ilvl > 0">
            <xsl:call-template name="repeat-string">
                <xsl:with-param name="string" select="$list-indent"/>
                <xsl:with-param name="count" select="$ilvl"/>
            </xsl:call-template>
        </xsl:if>
        <!-- Use unordered list format (- ) -->
        <xsl:text>- </xsl:text>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
    </xsl:template>
    
    <!-- Handle ListBullet paragraphs in table cells as unordered lists -->
    <xsl:template match="w:tc//w:p[w:pPr/w:pStyle[@w:val='ListBullet']]">
        <xsl:variable name="ilvl" select="number(w:pPr/w:numPr/w:ilvl/@w:val)"/>
        <!-- Add indentation based on list level (2 spaces per level) -->
        <xsl:if test="$ilvl > 0">
            <xsl:call-template name="repeat-string">
                <xsl:with-param name="string" select="$list-indent"/>
                <xsl:with-param name="count" select="$ilvl"/>
            </xsl:call-template>
        </xsl:if>
        <!-- Use unordered list format (- ) -->
        <xsl:text>- </xsl:text>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
    </xsl:template>
    
    <!-- Handle paragraphs that might be bullet lists (check for list style) - but not if they're headings or ListNumber -->
    <xsl:template match="w:p[w:pPr/w:pStyle[contains(@w:val, 'List') or contains(@w:val, 'Bullet')] and not(w:pPr/w:pStyle[@w:val='ListNumber']) and not(w:pPr/w:pStyle[@w:val='ListBullet']) and not(w:pPr/w:numPr)]">
        <xsl:text>

</xsl:text>
        <xsl:text>- </xsl:text>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
    </xsl:template>
    
    <!-- Helper template to repeat a string -->
    <xsl:template name="repeat-string">
        <xsl:param name="string"/>
        <xsl:param name="count" select="0"/>
        <xsl:if test="$count > 0">
            <xsl:value-of select="$string"/>
            <xsl:call-template name="repeat-string">
                <xsl:with-param name="string" select="$string"/>
                <xsl:with-param name="count" select="$count - 1"/>
            </xsl:call-template>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle caption paragraphs - minimal spacing to keep with table/figure -->
    <xsl:template match="w:p[w:pPr/w:pStyle[@w:val='Caption']]">
        <xsl:text>

</xsl:text>
        <xsl:text>**</xsl:text>
        <xsl:apply-templates select="node()[not(self::w:pPr)]"/>
        <xsl:text>**</xsl:text>
    </xsl:template>
    
    <!-- Handle HTMLPreformatted paragraphs - convert to Markdown code blocks -->
    <xsl:template match="w:p[w:pPr/w:pStyle[@w:val='HTMLPreformatted']]">
        <xsl:text>
</xsl:text>
        <xsl:text>```</xsl:text>
        <xsl:text>
</xsl:text>
        <!-- Process all content, converting w:br to newlines -->
        <xsl:for-each select="node()[not(self::w:pPr)]">
            <xsl:choose>
                <xsl:when test="self::w:r">
                    <!-- Text run - extract text content first -->
                    <xsl:apply-templates select="w:t"/>
                    <!-- Then handle line breaks if present -->
                    <xsl:if test="w:br">
                        <xsl:text>
</xsl:text>
                    </xsl:if>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:apply-templates select="."/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:for-each>
        <xsl:text>
</xsl:text>
        <xsl:text>```</xsl:text>
        <xsl:text>
</xsl:text>
    </xsl:template>
    
    <!-- Handle regular paragraphs - add two newlines before every paragraph -->
    <xsl:template match="w:p">
        <xsl:text>

</xsl:text>
        <xsl:apply-templates select="node()"/>
    </xsl:template>
    
    <!-- Handle hyperlinks - external links with r:id (but not w:anchor) -->
    <xsl:template match="w:hyperlink[@r:id and not(@w:anchor)]">
        <xsl:variable name="link-text">
            <xsl:apply-templates select="node()"/>
        </xsl:variable>
        <xsl:variable name="rel-id" select="normalize-space(@r:id)"/>
        <xsl:if test="string-length(normalize-space($link-text)) > 0">
            <!-- External link - use placeholder that will be resolved in Java code -->
            <xsl:text>[</xsl:text>
            <xsl:value-of select="$link-text"/>
            <xsl:text>](link-</xsl:text>
            <xsl:value-of select="$rel-id"/>
            <xsl:text>)</xsl:text>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle hyperlinks - internal links with w:anchor (takes priority) -->
    <xsl:template match="w:hyperlink[@w:anchor]">
        <xsl:variable name="link-text">
            <xsl:apply-templates select="node()"/>
        </xsl:variable>
        <xsl:variable name="anchor" select="normalize-space(@w:anchor)"/>
        <xsl:if test="string-length(normalize-space($link-text)) > 0">
            <!-- Internal link - convert anchor to markdown anchor format -->
            <xsl:text>[</xsl:text>
            <xsl:value-of select="$link-text"/>
            <xsl:text>](#</xsl:text>
            <xsl:value-of select="translate($anchor, ' ', '-')"/>
            <xsl:text>)</xsl:text>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle hyperlinks without r:id or w:anchor (fallback) -->
    <xsl:template match="w:hyperlink">
        <xsl:apply-templates select="node()"/>
    </xsl:template>
    
    <!-- Helper template to process REF field and return markdown link -->
    <xsl:template name="process-ref-field">
        <xsl:param name="paragraph"/>
        <xsl:variable name="instr-text" select="$paragraph/w:r/w:instrText[starts-with(normalize-space(.), 'REF ')][1]"/>
        <xsl:variable name="instr-content" select="normalize-space($instr-text)"/>
        <!-- Extract reference ID from "REF _Refd21e706 \h" format -->
        <xsl:variable name="ref-id">
            <xsl:choose>
                <xsl:when test="contains($instr-content, ' ')">
                    <xsl:variable name="after-ref" select="substring-after($instr-content, 'REF ')"/>
                    <xsl:choose>
                        <xsl:when test="contains($after-ref, ' ')">
                            <xsl:value-of select="substring-before($after-ref, ' ')"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$after-ref"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="substring-after($instr-content, 'REF ')"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        
        <!-- Extract display text: collect all text nodes between separate and end fldChar -->
        <xsl:variable name="display-text">
            <xsl:for-each select="$paragraph/w:r">
                <xsl:variable name="is-separate" select="w:fldChar[@w:fldCharType='separate']"/>
                <xsl:variable name="is-end" select="w:fldChar[@w:fldCharType='end']"/>
                <xsl:variable name="has-separate-before" select="preceding-sibling::w:r[w:fldChar[@w:fldCharType='separate']]"/>
                <xsl:variable name="has-end-before" select="preceding-sibling::w:r[w:fldChar[@w:fldCharType='end']]"/>
                <!-- Extract text if: after separate, before end, and not separate/end itself -->
                <xsl:if test="$has-separate-before and not($has-end-before) and not($is-end) and not($is-separate)">
                    <xsl:apply-templates select="w:t"/>
                </xsl:if>
            </xsl:for-each>
        </xsl:variable>
        
        <!-- Output as markdown link -->
        <xsl:if test="string-length(normalize-space($ref-id)) > 0 and string-length(normalize-space($display-text)) > 0">
            <xsl:text>[</xsl:text>
            <xsl:value-of select="normalize-space($display-text)"/>
            <xsl:text>](#</xsl:text>
            <xsl:value-of select="translate($ref-id, ' ', '-')"/>
            <xsl:text>)</xsl:text>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle text runs with formatting -->
    <xsl:template match="w:r">
        <!-- Check if this run is part of a field (between separate and end) -->
        <xsl:variable name="has-separate-before" select="preceding-sibling::w:r[w:fldChar[@w:fldCharType='separate']]"/>
        <xsl:variable name="has-end-before" select="preceding-sibling::w:r[w:fldChar[@w:fldCharType='end']]"/>
        <xsl:variable name="is-separate" select="w:fldChar[@w:fldCharType='separate']"/>
        <xsl:variable name="is-end" select="w:fldChar[@w:fldCharType='end']"/>
        <xsl:variable name="is-in-field" select="$has-separate-before and not($has-end-before) and not($is-separate) and not($is-end) and w:t"/>
        
        <!-- Check if this is a REF field by looking for REF instruction in the paragraph -->
        <xsl:variable name="is-ref-field">
            <xsl:if test="$is-in-field">
                <xsl:variable name="parent-paragraph" select="parent::w:p"/>
                <xsl:variable name="has-ref-instr" select="$parent-paragraph/w:r/w:instrText[starts-with(normalize-space(.), 'REF ')]"/>
                <xsl:value-of select="boolean($has-ref-instr)"/>
            </xsl:if>
        </xsl:variable>
        
        <!-- Check if this is the first text run in a REF field -->
        <xsl:variable name="is-first-ref-text">
            <xsl:if test="$is-in-field and $is-ref-field = 'true'">
                <xsl:value-of select="not(preceding-sibling::w:r[w:t and preceding-sibling::w:r[w:fldChar[@w:fldCharType='separate']] and not(preceding-sibling::w:r[w:fldChar[@w:fldCharType='end']])])"/>
            </xsl:if>
        </xsl:variable>
        
        <!-- If this is the first text run in a REF field, process the entire REF field -->
        <xsl:if test="$is-in-field and $is-ref-field = 'true' and $is-first-ref-text = 'true'">
            <xsl:call-template name="process-ref-field">
                <xsl:with-param name="paragraph" select="parent::w:p"/>
            </xsl:call-template>
        </xsl:if>
        
        <!-- Process text normally if:
             1. Not in any field, OR
             2. In a field but not a REF field (e.g., TITLE field) -->
        <xsl:if test="not($is-in-field) or ($is-in-field and $is-ref-field != 'true')">
            <!-- Check if this is an HTMLCode run in a non-HTMLPreformatted paragraph -->
            <xsl:variable name="is-html-code" select="w:rPr/w:rStyle[@w:val='HTMLCode'] and not(parent::w:p/w:pPr/w:pStyle[@w:val='HTMLPreformatted'])"/>
            <xsl:variable name="is-bold" select="w:rPr/w:b and (not(w:rPr/w:b/@w:val) or w:rPr/w:b/@w:val = 'true' or w:rPr/w:b/@w:val = '1')"/>
            <xsl:variable name="is-italic" select="w:rPr/w:i and (not(w:rPr/w:i/@w:val) or w:rPr/w:i/@w:val = 'true' or w:rPr/w:i/@w:val = '1')"/>
            <xsl:variable name="text-content">
                <xsl:apply-templates select="node()[not(self::w:rPr)]"/>
            </xsl:variable>
            <xsl:variable name="normalized-text" select="normalize-space($text-content)"/>
            <xsl:if test="string-length($normalized-text) > 0">
                <xsl:choose>
                    <!-- HTMLCode style or xml:space="preserve" takes precedence - convert to inline code -->
                    <xsl:when test="$is-html-code">
                        <xsl:text>`</xsl:text>
                        <xsl:value-of select="$text-content"/>
                        <xsl:text>`</xsl:text>
                    </xsl:when>
                    <xsl:when test="$is-bold and $is-italic">
                        <xsl:text>***</xsl:text>
                        <xsl:value-of select="$text-content"/>
                        <xsl:text>***</xsl:text>
                    </xsl:when>
                    <xsl:when test="$is-bold">
                        <xsl:text>**</xsl:text>
                        <xsl:value-of select="$text-content"/>
                        <xsl:text>**</xsl:text>
                    </xsl:when>
                    <xsl:when test="$is-italic">
                        <xsl:text>*</xsl:text>
                        <xsl:value-of select="$text-content"/>
                        <xsl:text>*</xsl:text>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:value-of select="$text-content"/>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle plain text nodes -->
    <xsl:template match="text()">
        <xsl:value-of select="."/>
    </xsl:template>
    
    <!-- Handle tables -->
    <xsl:template match="w:tbl">
        <xsl:variable name="rows" select="w:tr"/>
        <xsl:variable name="row-count" select="count($rows)"/>
        
        <!-- If table has only one row, output each cell as a separate paragraph -->
        <xsl:if test="$row-count = 1">
            <xsl:for-each select="$rows[1]/w:tc">
                <xsl:text>

</xsl:text>
                <xsl:variable name="cell-content">
                    <xsl:apply-templates select="node()"/>
                </xsl:variable>
                <xsl:value-of select="normalize-space($cell-content)"/>
            </xsl:for-each>
            <xsl:text>

</xsl:text>
        </xsl:if>
        
        <!-- If table has multiple rows, output as markdown table -->
        <xsl:if test="$row-count > 1">
            <xsl:text>

</xsl:text>
            <!-- Calculate max columns by finding the row with most cells -->
            <xsl:variable name="max-cols">
                <xsl:call-template name="max-columns">
                    <xsl:with-param name="rows" select="$rows"/>
                    <xsl:with-param name="max-so-far" select="0"/>
                </xsl:call-template>
            </xsl:variable>
            <xsl:variable name="max-cols-value" select="number($max-cols)"/>
            
            <!-- Process all rows -->
            <xsl:for-each select="$rows">
                <xsl:variable name="position" select="position()"/>
                <xsl:apply-templates select="." mode="table-row">
                    <xsl:with-param name="max-cols" select="$max-cols-value"/>
                </xsl:apply-templates>
                <!-- Add separator row after first row -->
                <xsl:if test="$position = 1">
                    <xsl:text>|</xsl:text>
                    <xsl:call-template name="repeat-string">
                        <xsl:with-param name="string" select="' --- |'"/>
                        <xsl:with-param name="count" select="$max-cols-value"/>
                    </xsl:call-template>
                    <xsl:text>
</xsl:text>
                </xsl:if>
            </xsl:for-each>
            <xsl:text>
</xsl:text>
        </xsl:if>
    </xsl:template>
    
    <!-- Helper template to find max columns in a table -->
    <xsl:template name="max-columns">
        <xsl:param name="rows"/>
        <xsl:param name="max-so-far" select="0"/>
        <xsl:choose>
            <xsl:when test="count($rows) = 0">
                <xsl:value-of select="$max-so-far"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="first-row-cols" select="count($rows[1]/w:tc)"/>
                <xsl:variable name="new-max">
                    <xsl:choose>
                        <xsl:when test="$first-row-cols > $max-so-far">
                            <xsl:value-of select="$first-row-cols"/>
                        </xsl:when>
                        <xsl:otherwise>
                            <xsl:value-of select="$max-so-far"/>
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:call-template name="max-columns">
                    <xsl:with-param name="rows" select="$rows[position() > 1]"/>
                    <xsl:with-param name="max-so-far" select="number($new-max)"/>
                </xsl:call-template>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Handle table rows -->
    <xsl:template match="w:tr" mode="table-row">
        <xsl:param name="max-cols" select="1"/>
        <xsl:text>|</xsl:text>
        <xsl:for-each select="w:tc">
            <xsl:text> </xsl:text>
            <xsl:variable name="cell-content">
                <xsl:apply-templates select="node()"/>
            </xsl:variable>
            <xsl:value-of select="normalize-space($cell-content)"/>
            <xsl:text> |</xsl:text>
        </xsl:for-each>
        <!-- Fill remaining columns if needed -->
        <xsl:variable name="current-cols" select="count(w:tc)"/>
        <xsl:if test="$current-cols &lt; $max-cols">
            <xsl:call-template name="repeat-string">
                <xsl:with-param name="string" select="'  |'"/>
                <xsl:with-param name="count" select="$max-cols - $current-cols"/>
            </xsl:call-template>
        </xsl:if>
        <xsl:text>
</xsl:text>
    </xsl:template>
    
    <!-- Handle table cells in normal flow (not in table mode) -->
    <xsl:template match="w:tc">
        <xsl:apply-templates select="node()"/>
    </xsl:template>
    
    <!-- Handle table rows in normal flow -->
    <xsl:template match="w:tr">
        <xsl:apply-templates select="node()"/>
    </xsl:template>
    
    <!-- Handle document and body elements -->
    <xsl:template match="w:document|w:body">
        <xsl:apply-templates select="node()"/>
    </xsl:template>
    
    <!-- Process drawing nodes to extract image placeholders -->
    <xsl:template match="w:drawing">
        <xsl:apply-templates select="node()"/>
    </xsl:template>
    
    <!-- Handle images: use old attach format if expandImages is true, otherwise use markdown -->
    <xsl:template match="a:blip[@r:embed]">
        <xsl:variable name="image-id" select="normalize-space(@r:embed)"/>
        <xsl:choose>
            <xsl:when test="$expandImages = 'true'">
                <!-- Old format: ${attach(...)} -->
                <xsl:value-of select="concat('${attach(', $image-id, ')}')"/>
            </xsl:when>
            <xsl:otherwise>
                <!-- New format: markdown image reference -->
                <xsl:text>

![Image](image-</xsl:text>
                <xsl:value-of select="$image-id"/>
                <xsl:text>.png)

</xsl:text>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>
    
    <!-- Handle bookmarks - create markdown anchor points for internal links -->
    <!-- Only create anchors for bookmarks that are actually referenced -->
    <xsl:template match="w:bookmarkStart">
        <xsl:variable name="bookmark-name" select="normalize-space(@w:name)"/>
        <xsl:if test="string-length($bookmark-name) > 0">
            <!-- Check if this bookmark is referenced by any hyperlink -->
            <xsl:variable name="is-referenced-by-hyperlink" 
                          select="boolean(//w:hyperlink[@w:anchor = $bookmark-name])"/>
            
            <!-- Check if this bookmark is referenced by any REF field -->
            <!-- Extract all REF field IDs and check if any matches this bookmark name -->
            <xsl:variable name="is-referenced-by-ref">
                <xsl:for-each select="//w:r[w:instrText[starts-with(normalize-space(.), 'REF ')]]">
                    <xsl:variable name="instr-content" select="normalize-space(w:instrText)"/>
                    <xsl:if test="string-length($instr-content) > 4">
                        <xsl:variable name="after-ref" select="substring-after($instr-content, 'REF ')"/>
                        <xsl:variable name="ref-id">
                            <xsl:choose>
                                <xsl:when test="contains($after-ref, ' ')">
                                    <xsl:value-of select="substring-before($after-ref, ' ')"/>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:value-of select="$after-ref"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:variable>
                        <xsl:if test="normalize-space($ref-id) = $bookmark-name">true</xsl:if>
                    </xsl:if>
                </xsl:for-each>
            </xsl:variable>
            
            <!-- Only create anchor if referenced -->
            <xsl:if test="$is-referenced-by-hyperlink or contains($is-referenced-by-ref, 'true')">
                <!-- Create HTML anchor for markdown compatibility -->
                <xsl:text>&lt;a id="</xsl:text>
                <xsl:value-of select="translate($bookmark-name, ' ', '-')"/>
                <xsl:text>"&gt;&lt;/a&gt;</xsl:text>
            </xsl:if>
        </xsl:if>
    </xsl:template>
    
    <!-- Handle SEQ fields (table/figure numbering) -->
    <xsl:template match="w:fldSimple[starts-with(normalize-space(@w:instr), 'SEQ ')]">
        <xsl:apply-templates select="w:r/w:t"/>
    </xsl:template>
    
    <!-- Ignore formatting and other unnecessary nodes -->
    <xsl:template match="w:pPr|w:rPr|w:proofErr|w:br|w:bookmarkEnd
        |w:sectPr|w:lastRenderedPageBreak
        |w:tblPr|w:tblGrid|w:trPr|w:tcPr|w:sheetViews|w:sheetView
        |w:sheetFormatPr|w:dimension|w:extLst|w:pageMargins|w:instrText|w:fldChar"/>
    
    <!-- Ignore TOC entries (both in body and in table cells) -->
    <xsl:template match="w:p[w:pPr/w:pStyle[starts-with(@w:val, 'TOC')]]"/>
    
    <!-- Default template for unmatched elements -->
    <xsl:template match="*">
        <xsl:apply-templates select="node()"/>
    </xsl:template>

</xsl:stylesheet>