/*
 * Decompiled with CFR 0.152.
 */
package com.oxygenxml.positron.core.util.attach.pdf;

import com.oxygenxml.positron.core.util.attach.DocumentSplitFolderUtils;
import com.oxygenxml.positron.core.util.attach.pdf.FontAwareTextStripper;
import com.oxygenxml.positron.core.util.attach.pdf.ImagePosition;
import com.oxygenxml.positron.core.util.attach.pdf.PDFImageExtractor;
import com.oxygenxml.positron.core.util.attach.pdf.PdfDocumentUtils;
import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import javax.imageio.ImageIO;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.graphics.image.PDImageXObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ro.sync.basic.io.FileSystemUtil;
import ro.sync.basic.util.URLUtil;

public class PdfSplitFolderManager {
    private static final Logger log = LoggerFactory.getLogger(PdfSplitFolderManager.class);
    private static final String PDF_SPLITS_FOLDER_NAME = "oxygen-ai-pdf-splits/";
    private static final String FOLDER_NAME_SEPARATOR = "-";
    private static final String SECTION_FILE_NAME_FORMAT = "%s-%03d.md";
    private static final String IMAGE_FILE_NAME_FORMAT = "image_%03d.png";
    private static final String MARKDOWN_HEADING_PREFIX = "# ";
    private static final String MARKDOWN_SUBHEADING_PREFIX = "## ";
    private static final String MARKDOWN_CODE_BLOCK_DELIMITER = "```";
    private static final String MARKDOWN_CODE_BLOCK_START = "```\n";
    private static final String MARKDOWN_IMAGE_PREFIX = "![Image from page ";
    private static final String PARAGRAPH_BREAK_MARKER = "<PARAGRAPH_BREAK>";
    private static final String DOUBLE_NEWLINE = "\n\n";
    private static final String SINGLE_NEWLINE = "\n";
    private static final String IMAGE_FORMAT_PNG = "png";
    private static final float IMAGE_POSITION_TOLERANCE = 100.0f;
    private static final int MIN_CONTENT_LENGTH = 10;
    private static final float DISTANCE_TOLERANCE = 10.0f;
    private static final String DEFAULT_SECTION_TITLE_INTRODUCTION = "Introduction";
    private static final String DEFAULT_SECTION_TITLE_CONTENT = "Content";
    private static final String INSTANTIATION_ERROR_MESSAGE = "Instantiation of this utility class is not allowed!";

    private PdfSplitFolderManager() {
        throw new UnsupportedOperationException(INSTANTIATION_ERROR_MESSAGE);
    }

    public static File getOrCreateSplitFolder(String pdfFilePath, byte[] pdfBytes) throws Exception {
        String folderName = URLUtil.extractFileName((String)pdfFilePath) + FOLDER_NAME_SEPARATOR + pdfFilePath.hashCode();
        File tempDir = FileSystemUtil.getOxygenTempDirectory();
        File splitFolder = new File(tempDir, PDF_SPLITS_FOLDER_NAME + folderName);
        if (DocumentSplitFolderUtils.isFolderValid(splitFolder, pdfBytes)) {
            return splitFolder;
        }
        splitFolder.mkdirs();
        splitFolder.deleteOnExit();
        File imagesDir = new File(splitFolder, "images");
        imagesDir.mkdirs();
        imagesDir.deleteOnExit();
        PdfSplitFolderManager.splitPdfIntoFiles(pdfFilePath, pdfBytes, splitFolder, imagesDir);
        DocumentSplitFolderUtils.saveMetadata(splitFolder, pdfBytes);
        return splitFolder;
    }

    private static void splitPdfIntoFiles(String pdfFilePath, byte[] pdfBytes, File splitFolder, File imagesDir) throws Exception {
        try (PDDocument pdDoc = PdfDocumentUtils.parsePDFDocument(pdfBytes);){
            Map<Integer, List<ImageInfo>> imagesByPage = PdfSplitFolderManager.extractImagesWithPageNumbers(pdDoc);
            Map<Integer, Float> pageHeights = PdfDocumentUtils.buildPageHeightMap(pdDoc);
            FontAwareTextStripper stripper = new FontAwareTextStripper();
            stripper.getText(pdDoc);
            List<PdfDocumentUtils.TextLineInfo> textLines = stripper.getTextLines();
            textLines = PdfDocumentUtils.filterTopMarginContent(textLines, pageHeights);
            textLines = PdfDocumentUtils.convertTablesToMarkdown(textLines);
            List<PdfSection> sections = PdfSplitFolderManager.splitTextIntoSections(textLines);
            HashMap<ImageInfo, String> savedImages = new HashMap<ImageInfo, String>();
            HashMap<ImageInfo, String> imageInfoToFilename = new HashMap<ImageInfo, String>();
            ArrayList<String> allImageFiles = new ArrayList<String>();
            PdfSplitFolderManager.saveImages(imagesDir, imagesByPage, savedImages, imageInfoToFilename, allImageFiles);
            PdfSplitFolderManager.saveSections(splitFolder, imagesByPage, textLines, sections, imageInfoToFilename);
            PdfSplitFolderManager.createTableOfContents(splitFolder, pdfFilePath, sections, allImageFiles);
        }
    }

    private static void saveSections(File splitFolder, Map<Integer, List<ImageInfo>> imagesByPage, List<PdfDocumentUtils.TextLineInfo> textLines, List<PdfSection> sections, Map<ImageInfo, String> imageInfoToFilename) throws IOException {
        int size = sections.size();
        for (int i = 0; i < size; ++i) {
            PdfSection section = sections.get(i);
            String fileName = String.format(SECTION_FILE_NAME_FORMAT, DocumentSplitFolderUtils.sanitizeFilename(section.title), i + 1);
            File sectionFile = new File(splitFolder, fileName);
            sectionFile.deleteOnExit();
            StringBuilder content = new StringBuilder();
            content.append(MARKDOWN_HEADING_PREFIX).append(section.title).append(DOUBLE_NEWLINE);
            content.append(PdfSplitFolderManager.insertImageReferences(section, imagesByPage, imageInfoToFilename, textLines));
            Files.write(sectionFile.toPath(), content.toString().getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
        }
    }

    private static void saveImages(File imagesDir, Map<Integer, List<ImageInfo>> imagesByPage, Map<ImageInfo, String> savedImages, Map<ImageInfo, String> imageInfoToFilename, List<String> allImageFiles) throws IOException {
        int imageIndex = 1;
        for (Map.Entry<Integer, List<ImageInfo>> entry : imagesByPage.entrySet()) {
            for (ImageInfo imageInfo : entry.getValue()) {
                String imageFileName;
                if (savedImages.containsKey(imageInfo)) {
                    imageFileName = savedImages.get(imageInfo);
                } else {
                    imageFileName = String.format(IMAGE_FILE_NAME_FORMAT, imageIndex++);
                    File imageFile = new File(imagesDir, imageFileName);
                    imageFile.deleteOnExit();
                    try (FileOutputStream fos = new FileOutputStream(imageFile);){
                        fos.write(imageInfo.imageBytes);
                    }
                    savedImages.put(imageInfo, imageFileName);
                    allImageFiles.add(imageFileName);
                }
                imageInfoToFilename.put(imageInfo, imageFileName);
            }
        }
    }

    private static Map<Integer, List<ImageInfo>> extractImagesWithPageNumbers(PDDocument document) throws Exception {
        LinkedHashMap<Integer, List<ImageInfo>> imagesByPage = new LinkedHashMap<Integer, List<ImageInfo>>();
        HashMap<Integer, ImageInfo> uniqueImages = new HashMap<Integer, ImageInfo>();
        int pageNum = 0;
        for (PDPage page : document.getPages()) {
            PDRectangle pageMediaBox = page.getMediaBox();
            float pageHeight = pageMediaBox != null ? pageMediaBox.getHeight() : 792.0f;
            PDFImageExtractor extractor = new PDFImageExtractor();
            extractor.processPage(page);
            ArrayList<ImageInfo> pageImages = new ArrayList<ImageInfo>();
            int pageNum1Based = pageNum + 1;
            for (ImagePosition imagePos : extractor.getImages()) {
                PdfSplitFolderManager.extractImage(uniqueImages, pageHeight, pageImages, pageNum1Based, imagePos);
            }
            if (!pageImages.isEmpty()) {
                imagesByPage.put(pageNum + 1, pageImages);
            }
            ++pageNum;
        }
        return imagesByPage;
    }

    private static void extractImage(Map<Integer, ImageInfo> uniqueImages, float pageHeight, List<ImageInfo> pageImages, int pageNum1Based, ImagePosition imagePos) {
        PDImageXObject imageObj = imagePos.image;
        float yPosPdf = imagePos.yPosition;
        float yPos = pageHeight - yPosPdf;
        try {
            BufferedImage image = imageObj.getImage();
            try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
                ImageInfo imageInfo;
                ImageIO.write((RenderedImage)image, IMAGE_FORMAT_PNG, baos);
                byte[] byteArray = baos.toByteArray();
                int imageHash = Arrays.hashCode(byteArray);
                if (uniqueImages.containsKey(imageHash)) {
                    imageInfo = uniqueImages.get(imageHash);
                } else {
                    imageInfo = new ImageInfo(byteArray, yPos, pageNum1Based);
                    uniqueImages.put(imageHash, imageInfo);
                }
                pageImages.add(imageInfo);
            }
        }
        catch (IOException ex) {
            log.warn((Throwable)ex);
        }
    }

    private static String insertImageReferences(PdfSection section, Map<Integer, List<ImageInfo>> imagesByPage, Map<ImageInfo, String> imageInfoToFilename, List<PdfDocumentUtils.TextLineInfo> textLines) {
        String trimmedContent = section.content.trim();
        if (trimmedContent.isEmpty() || trimmedContent.length() < 10) {
            return section.content;
        }
        Set<Integer> sectionPages = PdfSplitFolderManager.getUniquePagesFromSection(section);
        List<PdfDocumentUtils.TextLineInfo> sectionTextLines = PdfSplitFolderManager.extractSectionTextLines(section, textLines, sectionPages);
        HashMap<Integer, Float> minYByPage = new HashMap<Integer, Float>();
        HashMap<Integer, Float> maxYByPage = new HashMap<Integer, Float>();
        for (PdfDocumentUtils.TextLineInfo textLine : sectionTextLines) {
            int page = textLine.pageNumber;
            float yPos = textLine.yPosition;
            if (!(yPos > 0.0f)) continue;
            minYByPage.put(page, Float.valueOf(Math.min(minYByPage.getOrDefault(page, Float.valueOf(Float.MAX_VALUE)).floatValue(), yPos)));
            maxYByPage.put(page, Float.valueOf(Math.max(maxYByPage.getOrDefault(page, Float.valueOf(0.0f)).floatValue(), yPos)));
        }
        List<ImagePlacement> imagePlacements = PdfSplitFolderManager.computeImagePlacements(imagesByPage, imageInfoToFilename, sectionPages, minYByPage, maxYByPage);
        Map<ImagePlacement, PdfDocumentUtils.TextLineInfo> imageToTextLine = PdfSplitFolderManager.computeImageToTextLinesPlacement(sectionTextLines, imagePlacements);
        HashMap<PdfDocumentUtils.TextLineInfo, List<ImagePlacement>> textLineToImages = new HashMap<PdfDocumentUtils.TextLineInfo, List<ImagePlacement>>();
        for (Map.Entry<ImagePlacement, PdfDocumentUtils.TextLineInfo> entry : imageToTextLine.entrySet()) {
            PdfDocumentUtils.TextLineInfo textLine = entry.getValue();
            ImagePlacement placement = entry.getKey();
            textLineToImages.computeIfAbsent(textLine, k -> new ArrayList()).add(placement);
        }
        String[] lines = section.content.split(SINGLE_NEWLINE, -1);
        StringBuilder result = new StringBuilder();
        Set<ImageInfo> insertedImages = PdfSplitFolderManager.insertImageAfterEachLine(sectionTextLines, textLineToImages, lines, result);
        PdfSplitFolderManager.insertRemainingImages(imagePlacements, result, insertedImages);
        return result.toString();
    }

    private static List<ImagePlacement> computeImagePlacements(Map<Integer, List<ImageInfo>> imagesByPage, Map<ImageInfo, String> imageInfoToFilename, Set<Integer> sectionPages, Map<Integer, Float> minYByPage, Map<Integer, Float> maxYByPage) {
        ArrayList<ImagePlacement> imagePlacements = new ArrayList<ImagePlacement>();
        ArrayList<Integer> sortedSectionPages = new ArrayList<Integer>(sectionPages);
        int size = sortedSectionPages.size();
        for (int i = 0; i < size; ++i) {
            int pageNum = (Integer)sortedSectionPages.get(i);
            if (!imagesByPage.containsKey(pageNum)) continue;
            Float sectionMinY = minYByPage.get(pageNum);
            Float sectionMaxY = maxYByPage.get(pageNum);
            if (sectionMinY == null || sectionMaxY == null) continue;
            boolean sectionContinuesOnNextPages = i < size - 1;
            for (ImageInfo imageInfo : imagesByPage.get(pageNum)) {
                String filename;
                if (imageInfo.pageNumber != pageNum) continue;
                boolean belongsToSection = true;
                if (imageInfo.yPosition > 0.0f) {
                    if (imageInfo.yPosition < sectionMinY.floatValue() - 100.0f) {
                        belongsToSection = false;
                    } else if (imageInfo.yPosition > sectionMaxY.floatValue() + 100.0f && !sectionContinuesOnNextPages) {
                        belongsToSection = false;
                    }
                }
                if (!belongsToSection || (filename = imageInfoToFilename.get(imageInfo)) == null) continue;
                imagePlacements.add(new ImagePlacement(imageInfo, filename, pageNum));
            }
        }
        imagePlacements.sort((a, b) -> {
            int pageCmp = Integer.compare(a.pageNumber, b.pageNumber);
            if (pageCmp != 0) {
                return pageCmp;
            }
            return Float.compare(a.imageInfo.yPosition, b.imageInfo.yPosition);
        });
        return imagePlacements;
    }

    private static Map<ImagePlacement, PdfDocumentUtils.TextLineInfo> computeImageToTextLinesPlacement(List<PdfDocumentUtils.TextLineInfo> sectionTextLines, List<ImagePlacement> imagePlacements) {
        HashMap<ImagePlacement, PdfDocumentUtils.TextLineInfo> imageToTextLine = new HashMap<ImagePlacement, PdfDocumentUtils.TextLineInfo>();
        for (ImagePlacement placement : imagePlacements) {
            PdfDocumentUtils.TextLineInfo closestTextLine = null;
            float minDistance = Float.MAX_VALUE;
            for (PdfDocumentUtils.TextLineInfo textLine : sectionTextLines) {
                float distance;
                if (textLine.pageNumber != placement.pageNumber || !((distance = Math.abs(textLine.yPosition - placement.imageInfo.yPosition)) < minDistance) && (!(distance < minDistance + 10.0f) || !(textLine.yPosition < placement.imageInfo.yPosition))) continue;
                minDistance = distance;
                closestTextLine = textLine;
            }
            if (closestTextLine == null) continue;
            imageToTextLine.put(placement, closestTextLine);
        }
        return imageToTextLine;
    }

    private static void insertRemainingImages(List<ImagePlacement> imagePlacements, StringBuilder result, Set<ImageInfo> insertedImages) {
        for (ImagePlacement placement : imagePlacements) {
            if (insertedImages.contains(placement.imageInfo)) continue;
            result.append(DOUBLE_NEWLINE);
            result.append(MARKDOWN_IMAGE_PREFIX).append(placement.pageNumber).append("](").append("images").append("/").append(placement.filename).append(")\n");
            insertedImages.add(placement.imageInfo);
        }
    }

    private static Set<ImageInfo> insertImageAfterEachLine(List<PdfDocumentUtils.TextLineInfo> sectionTextLines, Map<PdfDocumentUtils.TextLineInfo, List<ImagePlacement>> textLineToImages, String[] lines, StringBuilder result) {
        HashSet<ImageInfo> insertedImages = new HashSet<ImageInfo>();
        for (int i = 0; i < lines.length; ++i) {
            String line = lines[i];
            result.append(line);
            String lineText = line.trim();
            if (!(lineText.isEmpty() || lineText.startsWith("#") || lineText.startsWith("|"))) {
                for (PdfDocumentUtils.TextLineInfo textLine : sectionTextLines) {
                    String textLineText = textLine.text.trim();
                    if (!PdfSplitFolderManager.isTextLineMatch(lineText, textLineText)) continue;
                    List<ImagePlacement> imagesToInsert = textLineToImages.get(textLine);
                    if (imagesToInsert == null || imagesToInsert.isEmpty()) break;
                    imagesToInsert.sort((a, b) -> Float.compare(a.imageInfo.yPosition, b.imageInfo.yPosition));
                    PdfSplitFolderManager.insertRemainingImages(imagesToInsert, result, insertedImages);
                    break;
                }
            }
            if (i >= lines.length - 1) continue;
            result.append(SINGLE_NEWLINE);
        }
        return insertedImages;
    }

    private static boolean isTextLineMatch(String firstLineText, String secondLineText) {
        return secondLineText.contains(firstLineText) || firstLineText.contains(secondLineText) || PdfDocumentUtils.normalizeForMatching(secondLineText).equals(PdfDocumentUtils.normalizeForMatching(firstLineText));
    }

    private static Set<Integer> getUniquePagesFromSection(PdfSection section) {
        TreeSet<Integer> sectionPages = new TreeSet<Integer>();
        for (int page : section.pages) {
            sectionPages.add(page);
        }
        return sectionPages;
    }

    private static List<PdfDocumentUtils.TextLineInfo> extractSectionTextLines(PdfSection section, List<PdfDocumentUtils.TextLineInfo> textLines, Set<Integer> sectionPages) {
        ArrayList<PdfDocumentUtils.TextLineInfo> sectionTextLines = new ArrayList<PdfDocumentUtils.TextLineInfo>();
        String normalizedSectionContent = PdfDocumentUtils.normalizeForMatching(section.content);
        for (PdfDocumentUtils.TextLineInfo textLine : textLines) {
            String normalizedTextLine;
            if (!sectionPages.contains(textLine.pageNumber) || (normalizedTextLine = PdfDocumentUtils.normalizeForMatching(textLine.text)).isEmpty() || !normalizedSectionContent.contains(normalizedTextLine)) continue;
            sectionTextLines.add(textLine);
        }
        return sectionTextLines;
    }

    private static String normalizeParagraphSpacing(String content) {
        if (content == null || content.isEmpty()) {
            return content;
        }
        content = content.replace("\n<PARAGRAPH_BREAK>\n", DOUBLE_NEWLINE);
        return content;
    }

    private static List<PdfSection> splitTextIntoSections(List<PdfDocumentUtils.TextLineInfo> textLines) {
        ArrayList<PdfSection> sections = new ArrayList<PdfSection>();
        if (textLines.isEmpty()) {
            sections.add(new PdfSection(DEFAULT_SECTION_TITLE_CONTENT, "", new ArrayList<Integer>()));
            return sections;
        }
        List<PdfDocumentUtils.TocEntry> tocEntries = PdfDocumentUtils.extractTableOfContents(textLines);
        float avgFontSize = PdfDocumentUtils.calculateAverageFontSize(textLines);
        float maxFontSize = PdfDocumentUtils.calculateMaxFontSize(textLines);
        StringBuilder currentContent = new StringBuilder();
        String currentTitle = DEFAULT_SECTION_TITLE_INTRODUCTION;
        ArrayList<Integer> currentPages = new ArrayList<Integer>();
        boolean hasFoundFirstHeading = false;
        PdfDocumentUtils.TextLineInfo previousLineInfo = null;
        boolean inCodeBlock = false;
        StringBuilder codeBlockContent = new StringBuilder();
        for (int i = 0; i < textLines.size(); ++i) {
            Object formattedText;
            PdfDocumentUtils.TextLineInfo lineInfo = textLines.get(i);
            String trimmed = lineInfo.text.trim();
            boolean isMonospaced = PdfDocumentUtils.isLineMonospaced(lineInfo);
            if (trimmed.isEmpty()) {
                if (inCodeBlock) {
                    codeBlockContent.append(SINGLE_NEWLINE);
                    continue;
                }
                previousLineInfo = null;
                continue;
            }
            if (PdfDocumentUtils.isTableOfContentsLine(trimmed)) continue;
            boolean isHeading = PdfDocumentUtils.isLikelyHeading(lineInfo, avgFontSize, maxFontSize, i, textLines, tocEntries);
            boolean isSubsection = PdfDocumentUtils.isSubsectionHeading(trimmed);
            if (isHeading && !isSubsection) {
                if (inCodeBlock) {
                    currentContent.append(MARKDOWN_CODE_BLOCK_START).append(codeBlockContent.toString()).append(MARKDOWN_CODE_BLOCK_DELIMITER).append(DOUBLE_NEWLINE);
                    codeBlockContent = new StringBuilder();
                    inCodeBlock = false;
                }
                if (hasFoundFirstHeading && currentContent.length() > 0) {
                    sections.add(new PdfSection(currentTitle, PdfSplitFolderManager.normalizeParagraphSpacing(currentContent.toString()), new ArrayList<Integer>(currentPages)));
                    currentContent = new StringBuilder();
                    currentPages = new ArrayList();
                }
                currentTitle = trimmed;
                hasFoundFirstHeading = true;
                previousLineInfo = null;
                continue;
            }
            if (isSubsection && isHeading) {
                if (!currentPages.contains(lineInfo.pageNumber)) {
                    currentPages.add(lineInfo.pageNumber);
                }
                if (previousLineInfo != null) {
                    currentContent.append(SINGLE_NEWLINE);
                }
                formattedText = MARKDOWN_SUBHEADING_PREFIX + trimmed + DOUBLE_NEWLINE;
                currentContent.append((String)formattedText);
                previousLineInfo = null;
                continue;
            }
            if (!currentPages.contains(lineInfo.pageNumber)) {
                currentPages.add(lineInfo.pageNumber);
            }
            if (isMonospaced) {
                if (!inCodeBlock) {
                    if (previousLineInfo != null && previousLineInfo.text.trim().length() > 0) {
                        currentContent.append(DOUBLE_NEWLINE);
                    }
                    inCodeBlock = true;
                    codeBlockContent = new StringBuilder();
                }
                if (codeBlockContent.length() > 0) {
                    codeBlockContent.append(SINGLE_NEWLINE);
                }
                codeBlockContent.append(lineInfo.text);
                previousLineInfo = lineInfo;
                continue;
            }
            if (inCodeBlock) {
                currentContent.append(MARKDOWN_CODE_BLOCK_START).append(codeBlockContent.toString()).append(MARKDOWN_CODE_BLOCK_DELIMITER).append(DOUBLE_NEWLINE);
                codeBlockContent = new StringBuilder();
                inCodeBlock = false;
            }
            if (((String)(formattedText = PdfDocumentUtils.formatTextWithMarkdown(lineInfo))).endsWith(SINGLE_NEWLINE)) {
                formattedText = ((String)formattedText).substring(0, ((String)formattedText).length() - 1);
            }
            boolean isParagraphBreak = false;
            if (previousLineInfo != null) {
                isParagraphBreak = PdfDocumentUtils.hasVisualParagraphBreak(lineInfo, previousLineInfo);
            }
            if (isParagraphBreak && currentContent.length() > 0) {
                currentContent.append(SINGLE_NEWLINE).append(PARAGRAPH_BREAK_MARKER).append(SINGLE_NEWLINE);
            } else if (previousLineInfo != null && currentContent.length() > 0) {
                currentContent.append(SINGLE_NEWLINE);
            }
            currentContent.append((String)formattedText);
            previousLineInfo = lineInfo;
        }
        if (inCodeBlock) {
            currentContent.append(MARKDOWN_CODE_BLOCK_START).append(codeBlockContent.toString()).append(MARKDOWN_CODE_BLOCK_DELIMITER).append(SINGLE_NEWLINE);
        }
        if (currentContent.length() > 0) {
            sections.add(new PdfSection(currentTitle, PdfSplitFolderManager.normalizeParagraphSpacing(currentContent.toString()), new ArrayList<Integer>(currentPages)));
        }
        PdfSplitFolderManager.addEntireContentToUniqueSection(textLines, sections);
        List<PdfSection> filteredSections = PdfSplitFolderManager.filterEmptySections(sections);
        if (filteredSections.isEmpty() && !sections.isEmpty()) {
            filteredSections.add((PdfSection)sections.get(0));
        }
        return filteredSections;
    }

    private static List<PdfSection> filterEmptySections(List<PdfSection> sections) {
        ArrayList<PdfSection> filteredSections = new ArrayList<PdfSection>();
        for (int i = 0; i < sections.size(); ++i) {
            PdfSection section = sections.get(i);
            String trimmedContent = section.content.trim();
            if (!trimmedContent.isEmpty()) {
                filteredSections.add(section);
                continue;
            }
            if (i >= sections.size() - 1) continue;
            PdfSection nextSection = sections.get(i + 1);
            for (int page : section.pages) {
                if (nextSection.pages.contains(page)) continue;
                nextSection.pages.add(page);
            }
        }
        return filteredSections;
    }

    private static void addEntireContentToUniqueSection(List<PdfDocumentUtils.TextLineInfo> textLines, List<PdfSection> sections) {
        if (sections.isEmpty()) {
            StringBuilder allText = new StringBuilder();
            ArrayList<Integer> allPages = new ArrayList<Integer>();
            for (PdfDocumentUtils.TextLineInfo line : textLines) {
                String formattedText = PdfDocumentUtils.formatTextWithMarkdown(line);
                allText.append(formattedText).append(SINGLE_NEWLINE);
                if (allPages.contains(line.pageNumber)) continue;
                allPages.add(line.pageNumber);
            }
            sections.add(new PdfSection(DEFAULT_SECTION_TITLE_CONTENT, allText.toString(), allPages));
        }
    }

    private static void createTableOfContents(File splitFolder, String pdfFilePath, List<PdfSection> sections, List<String> imageFiles) throws IOException {
        StringBuilder toc = new StringBuilder();
        toc.append("# Table of Contents\n\n");
        toc.append("**Original PDF:** `").append(URLUtil.clearUserInfo((String)pdfFilePath)).append("`\n\n");
        toc.append("## Sections\n\n");
        for (int i = 0; i < sections.size(); ++i) {
            PdfSection section = sections.get(i);
            String fileName = String.format(SECTION_FILE_NAME_FORMAT, DocumentSplitFolderUtils.sanitizeFilename(section.title), i + 1);
            toc.append(String.format("%d. [%s](%s)\n", i + 1, section.title, fileName));
        }
        if (!imageFiles.isEmpty()) {
            toc.append("\n## Images\n\n");
            toc.append("The following images were extracted from the PDF:\n\n");
            for (String imageFile : imageFiles) {
                toc.append(String.format("- `%s/%s`\n", "images", imageFile));
            }
        }
        toc.append("\n## Instructions\n\n");
        toc.append("To process this PDF:\n\n");
        toc.append("1. Read the relevant section files using the `list_dir` and file reading tools\n");
        toc.append("2. Use `invoke_ai_agent` to process each section as needed\n");
        toc.append("3. Copy images from the `images/` directory to your documentation project as needed\n");
        File tocFile = new File(splitFolder, "table_of_contents.md");
        tocFile.deleteOnExit();
        Files.write(tocFile.toPath(), toc.toString().getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
    }

    private static class PdfSection {
        String title;
        String content;
        List<Integer> pages;

        PdfSection(String title, String content, List<Integer> pages) {
            this.title = title;
            this.content = content;
            this.pages = pages != null ? pages : new ArrayList();
        }

        public String toString() {
            return "Section " + this.title + " content " + this.content;
        }
    }

    private static class ImageInfo {
        byte[] imageBytes;
        float yPosition;
        int pageNumber;

        ImageInfo(byte[] imageBytes, float yPosition, int pageNumber) {
            this.imageBytes = imageBytes;
            this.yPosition = yPosition;
            this.pageNumber = pageNumber;
        }
    }

    private static class ImagePlacement {
        ImageInfo imageInfo;
        String filename;
        int pageNumber;

        ImagePlacement(ImageInfo imageInfo, String filename, int pageNumber) {
            this.imageInfo = imageInfo;
            this.filename = filename;
            this.pageNumber = pageNumber;
        }
    }
}

