/*
 * Decompiled with CFR 0.152.
 */
package com.oxygenxml.positron.connector.openai;

import com.azure.core.credential.AccessToken;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.oxygenxml.positron.connector.auth.OAuthCodeFlowHelper;
import com.oxygenxml.positron.connector.openai.OffsetDateTimeModule;
import java.io.IOException;
import java.time.OffsetDateTime;
import java.util.concurrent.Callable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OAuthAccessTokenCache {
    private static final Logger log = LoggerFactory.getLogger(OAuthAccessTokenCache.class);
    private static final long EXPIRATION_BUFFER_MINUTES = 5L;
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private final String serviceName;
    private final Callable<AccessToken> tokenProvider;
    private final OAuthCodeFlowHelper oauthHelper;
    private volatile AccessToken cachedToken;

    public OAuthAccessTokenCache(String serviceName, Callable<AccessToken> tokenProvider, OAuthCodeFlowHelper oauthHelper) {
        this.serviceName = serviceName;
        this.tokenProvider = tokenProvider;
        this.oauthHelper = oauthHelper;
        OBJECT_MAPPER.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        OBJECT_MAPPER.registerModule((Module)new OffsetDateTimeModule());
        AccessToken persistedToken = this.loadTokenFromStorage();
        if (persistedToken != null && !this.isTokenExpired(persistedToken)) {
            this.cachedToken = persistedToken;
        }
    }

    public String getTokenSync() throws IOException {
        try {
            if (this.cachedToken != null && !this.isTokenExpired(this.cachedToken)) {
                log.debug("Using cached access token for service: {}", (Object)this.serviceName);
                return this.cachedToken.getToken();
            }
            log.debug("Acquiring new access token for service: {}", (Object)this.serviceName);
            AccessToken newToken = this.tokenProvider.call();
            if (newToken == null) {
                throw new IOException("Cannot extract a valid token");
            }
            this.cachedToken = newToken;
            this.saveTokenToStorage(newToken);
            log.debug("Successfully acquired and cached new access token for service: {}", (Object)this.serviceName);
            return newToken.getToken();
        }
        catch (Exception e) {
            log.error("Failed to get access token for service {}: {}", new Object[]{this.serviceName, e.getMessage(), e});
            throw new IOException("Failed to get access token: " + e.getMessage(), e);
        }
    }

    public void clearCache() {
        log.debug("Clearing access token cache for service: {}", (Object)this.serviceName);
        this.cachedToken = null;
        if (this.oauthHelper != null) {
            this.oauthHelper.clearAccessToken(this.serviceName);
        }
    }

    private boolean isTokenExpired(AccessToken token) {
        if (token.getExpiresAt() == null) {
            return false;
        }
        OffsetDateTime now = OffsetDateTime.now();
        OffsetDateTime expirationWithBuffer = token.getExpiresAt().minusMinutes(5L);
        return now.isAfter(expirationWithBuffer);
    }

    private AccessToken loadTokenFromStorage() {
        if (this.oauthHelper == null) {
            return null;
        }
        try {
            String tokenJson = this.oauthHelper.loadAccessToken(this.serviceName);
            if (tokenJson == null || tokenJson.trim().isEmpty()) {
                return null;
            }
            SerializableAccessToken serializable = (SerializableAccessToken)OBJECT_MAPPER.readValue(tokenJson, SerializableAccessToken.class);
            return new AccessToken(serializable.getToken(), serializable.getExpiresAt());
        }
        catch (Exception e) {
            log.debug("Failed to load access token from storage for service {}: {}", (Object)this.serviceName, (Object)e.getMessage());
            return null;
        }
    }

    private void saveTokenToStorage(AccessToken token) {
        if (this.oauthHelper == null) {
            return;
        }
        try {
            SerializableAccessToken serializable = new SerializableAccessToken(token.getToken(), token.getExpiresAt());
            String tokenJson = OBJECT_MAPPER.writeValueAsString((Object)serializable);
            this.oauthHelper.saveAccessToken(this.serviceName, tokenJson);
        }
        catch (Exception e) {
            log.debug("Failed to save access token to storage for service {}: {}", (Object)this.serviceName, (Object)e.getMessage());
        }
    }

    static class SerializableAccessToken {
        @JsonProperty(value="token")
        private final String token;
        @JsonProperty(value="expiresAt")
        private final OffsetDateTime expiresAt;

        @JsonCreator
        public SerializableAccessToken(@JsonProperty(value="token") String token, @JsonProperty(value="expiresAt") OffsetDateTime expiresAt) {
            this.token = token;
            this.expiresAt = expiresAt;
        }

        public String getToken() {
            return this.token;
        }

        public OffsetDateTime getExpiresAt() {
            return this.expiresAt;
        }
    }
}

